<?php

namespace app\models;

use Yii;

/**
 * This is the model class for table "proveedores".
 *
 * @property int $prov_id
 * @property string $prov_identificacion
 * @property string $prov_nombre
 * @property string $prov_direccion
 * @property string $prov_correo
 * @property string $prov_telefono
 * @property string $prov_estado
 * @property int $created
 * @property string $created_at
 * @property string $updated_at
 * @property int $created_by
 * @property int $updated_by
 */
class Proveedores extends \yii\db\ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return 'proveedores';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['prov_identificacion', 'prov_nombre', 'prov_direccion',  'prov_telefono', 'prov_estado', 'created_at', 'updated_at', 'created_by', 'updated_by', 'prov_tipo_identificacion'], 'required'],
            [['created_by', 'updated_by'], 'integer'],
            [['created_at', 'updated_at'], 'safe'],
            [['prov_identificacion'], 'string', 'max' => 13],
            [['prov_identificacion'], 'match', 'pattern' => '/^\d+$/', 'message' => 'La identificación solo puede contener dígitos.'],
            [['prov_identificacion'], 'validateIdentificacion'],
            [['prov_nombre'], 'string', 'max' => 150],
            [['prov_nombre'], 'match', 'pattern' => '/^[\p{L}\s]+$/u', 'message' => 'El nombre solo puede contener letras'],
            [['prov_direccion'], 'string', 'max' => 300],
            [['prov_correo'], 'email'],
            [['prov_telefono'], 'string', 'max' => 30],
            [['prov_telefono'], 'match', 'pattern' => '/^\d+$/', 'message' => 'El teléfono solo puede contener dígitos.'],
            [['prov_estado'], 'string', 'max' => 1],
        ];
    }



    public function validateIdentificacion($attribute, $params)
    {
        if ($this->prov_tipo_identificacion == Parametros::CEDULA) {
            // Validar cédula
            $this->validateCedula($attribute, $params);
        } elseif ($this->prov_tipo_identificacion == Parametros::RUC) {
            // Validar RUC
            $this->validateRuc($attribute, $params);
        }
    }

    public function validateCedula($attribute, $params)
    {
        $cedula = $this->$attribute;

        if (!$this->isValidCedula($cedula)) {
            $this->addError($attribute, 'La cédula ingresada no es válida.');
        }
    }

    public function validateRuc($attribute, $params)
    {
        $ruc = $this->$attribute;

        // Verifica que el número tenga 13 dígitos
        if (strlen($ruc) != 13) {
            $this->addError($attribute, 'El RUC debe tener 13 dígitos.');
            return;
        }

        // Los primeros 10 dígitos deben ser una cédula válida
        $cedula = substr($ruc, 0, 10);
        if (!$this->isValidCedula($cedula)) {
            $this->addError($attribute, 'Los primeros diez dígitos del RUC no corresponden a una cédula válida.');
            return;
        }

        // Los últimos 3 dígitos deben estar en el rango de 001 a 999
        $sufijo = substr($ruc, 10, 3);
        if ($sufijo < '001' || $sufijo > '999') {
            $this->addError($attribute, 'Los últimos tres dígitos del RUC no son válidos.');
        }
    }

    private function isValidCedula($cedula)
    {
        // Verifica que el número tenga 10 dígitos
        if (strlen($cedula) != 10) {
            return false;
        }

        // Verifica que los dos primeros dígitos correspondan a una provincia válida (01-24)
        $provincia = substr($cedula, 0, 2);
        if ($provincia < 1 || $provincia > 24) {
            return false;
        }

        // Verifica que el tercer dígito sea menor que 6 (cédula de persona natural)
        $tercerDigito = $cedula[2];
        if ($tercerDigito >= 6) {
            return false;
        }

        // Algoritmo de validación de cédula
        $coeficientes = [2, 1, 2, 1, 2, 1, 2, 1, 2];
        $suma = 0;

        for ($i = 0; $i < 9; $i++) {
            $valor = $cedula[$i] * $coeficientes[$i];
            if ($valor >= 10) {
                $valor -= 9;
            }
            $suma += $valor;
        }

        $digitoVerificador = (10 - ($suma % 10)) % 10;

        return $cedula[9] == $digitoVerificador;
    }


    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'prov_id' => 'Prov ID',
            'prov_identificacion' => 'Identificación',
            'prov_nombre' => 'Proveedor',
            'prov_direccion' => 'Dirección',
            'prov_tipo_identificacion' => 'Tipo de identificación',
            'prov_correo' => 'Correo',
            'prov_telefono' => 'Teléfono',
            'prov_estado' => 'Estado',
            'created_at' => 'Fecha creación',
            'updated_at' => 'Fecha actualización',
            'created_by' => 'Creado por',
            'updated_by' => 'Actualizado por',
        ];
    }
}
